remark
\
It is suggested you view this tutorial in a maximized window
if you are using SQLTalk for Windows. To execute this 
tutorial, connect to the ISLAND database using SQLTalk and
issue the run command from the prompt: SQL> RUN SP.SQL;
/
pause;

remark
\
This is the SQLBase Procedure tutorial which can be run using SQLTalk. Many
of the following Procedure examples can be found in the SQLBase SQL Language
Reference Guide. It is recommended that this guide be read first before
running this tutorial. Three additional tutorials are also provided with your 
installation: TUTORIAL - an overview of SQL, TRIGGERS.SQL - a Triggers 
tutorial - and SP.APP which demonstrates how to access SQLBase Stored 
Procedures from SQLWindows.
This tutorial is organized in the following sections:
	. SECTION 1: NON-STORED PROCEDURE EXAMPLES
	. SECTION 2: STORED PROCEDURE EXAMPLES
	. SECTION 3: NESTED PROCEDURE EXAMPLES
	. SECTION 4: REQUIRED DATA AND DATABASE OBJECTS FOR SP.APP

The concluding section of this tutorial provides all of the database tables,
data and Stored Procedures required to run the SQLWindows sample application
SP.APP. Running either this entire script or SECTION 4 while connected to 
the ISLAND database will allow you to run SP.APP from SQLWindows. 
/
pause;

remark
\
=============================================================================
SECTION 1: NON-STORED PROCEDURE EXAMPLES
=============================================================================
/
-- Create the table required for some of the following Procedures.
create table GUEST_WT (NAME varchar(25), WEIGHT decimal (3,0), WHEN date);
insert into GUEST_WT values (:1,:2,:3)
\
JANE FYUDAI,132,13-OCT-1992
DON JACKSON,165,11-OCT-1992
DON JACKSON,159,12-OCT-1992
/
commit;

Remark
\
This is an example of a simple Procedure. In general, a Procedure is composed
of a name and sections labeled as Parameters, Local Variables and Actions. 
Local Variables and Parameters sections are optional if they are not
required. Depending on the design of the Procedure, various On Procedure 
<states> can be coded within the Actions section.  Procedures can be 
implemented as Stored (stored in the database) or Non-Stored as are the 
examples in this section.

Parameters are defined as they are in SAL - either as a standard variable 
parameter used as input only or as a Receive parameter. Receive parameters
can act as either input or output or both as in this case. Procedures do not
necessarily have to perform database operations as demonstrated in this
example.
/
pause;

procedure: ADD_TWO
Parameters
	Receive Number: nInOut
Local Variables
	Number: nVar1
Actions
	Set nVar1 = 5
	Set nInOut = nInOut + nVar1
\
5
/

Remark
\
Procedure language constructs and database function calls are virtually
identical to the SQLWindows SAL language. In this instance we will
connect to the database and insert a new record into the table GUEST_WT.
The Procedure database connection is to the same database of the calling
program's current cursor. Since there is no input to or output from the 
Procedure, there is no need to code a Parameters section.
/
procedure: GUEST_WT_ADD
Local Variables
	Sql Handle: hSqlCur1
Actions
	Call SqlConnect( hSqlCur1 )
	Call SqlImmediate( 'Insert into GUEST_WT \
			values (\'CLINT WESTWOOD\',180,1988-06-25)')
;

Remark
\
The calling program can either commit or rollback any previous uncommitted
Procedures. In this case we will rollback the insert operation of the 
previous Procedure.
/
pause;
rollback;

Remark
\
In many instances, it is NOT necessary to provide the On Procedure <states> 
for a Procedure.  The default state - On Procedure Execute - is often 
adequate for many Procedures.  However, there are (2) scenarios which do 
require 1 or more On Procedure <states> to be explictly coded.
	. If you wish to repeatedly execute a Procedure and  - as in the case
	  of supplying different parameter values - it is more efficient
	  to code an On Procedure Startup state where commands that require
	  only a single execution can be isolated to avoid unnecessary 
	  re-execution. This would include database connections,
	  variable assignments and compilation of SQL statements.
	. In cases, where more than a single row is to be fetched, it is 
	  essential to have an On Procedure Fetch state.
The below example demonstates both of the above scenarios:
/
pause;

-- Compile the Procedure
prepare
procedure: GUEST_WTPROC
Parameters
	String: sName
	Receive Number: nWeight
Local Variables
	Sql Handle: hSqlCur1
	String: sSelect 
	Number: nInd
Actions
	! This state is processed only once - on the 1st EXECUTE by the
	! caller. In cases of repeated executes from the caller, this 
	! provides a section for commands which do NOT require repeated 
	! processing and as a result can reduce the overhead of the 
	! Procedure.
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		Set sSelect = 'Select WEIGHT from GUEST_WT \
				where NAME = :sName into :nWeight'
		Call SqlPrepare(hSqlCur1, sSelect)

	! This state is processed on every EXECUTE issued by the caller.
	! If no states are coded, the Procedure defaults to this state.
	On Procedure Execute
		Call SqlExecute(hSqlCur1)

	! This state is processed on every FETCH issued by the caller and
	! is essential to fetching multiple row queries. 
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Return 0

	! This state is processed only after the Procedure has finished all
	! processing and only if another command is compiled or executed on 
	! the caller's current cursor or that cursor becomes disconnected.
	On Procedure Close
		Call SqlDisconnect(hSqlCur1)
/

Remark
\
First execution of the Procedure by the caller.  On Procedure Startup and 
On Procedure Execute states processed. Note that the second comma found
in the binding of the Procedure using the SQLTalk Perform command provides
the required placeholder for the Procedure Receive Parameter nWeight. 
All Procedure parameters must be supplied either an argument value or a 
positional placeholder.
/
pause;

perform GUEST_WTPROC
\
JANE FYUDAI,,
/

-- First fetch by caller. On Procedure Fetch state processed.
fetch 1;

Remark
\
2nd execution by caller with a different bind value. Only the On Procedure 
Execute state processed.
/
perform GUEST_WTPROC
\
DON JACKSON,,
/

Remark
\
Another fetch by the caller - this time with a different bind value.
On Procedure Fetch processed 2 times.
/
fetch 2;

-- On Procedure Close state processed for previous procedure.
select * from GUEST_WT;

Remark
\
In this example, a compile, execute and fetch of a single row from a
Procedure with no explicit On <states> will be demonstrated.  The Procedure
defaults to the On Procedure Execute state for all code under Actions.
/
pause;

procedure: GUEST_WTPROC
Parameters
	Receive Number: nSumWeight
Local Variables
	Sql Handle: hSqlCur1
	String: sSelect 
	Number: nInd
Actions
	! No On states explicitly coded, hence we default to the On Procedure 
	! Execute state.
	Call SqlConnect(hSqlCur1)
	! Aggregate queries are appropiate for this example.
	Set sSelect = 'Select max(WEIGHT) from GUEST_WT	into :nSumWeight'
	Call SqlPrepare(hSqlCur1, sSelect)
	Call SqlExecute(hSqlCur1)

	! In this case the On Procedure Fetch state is NOT coded. This makes
	! it possible for the caller to FETCH from the ON Procedure Execute 
	! state by embedding SAL fetch calls like SqlFetchNext( ).
	! However, in this instance you can only fetch and return to the 
	! caller a single row (even if within the Procedure the fetch is in
	! a loop).  In this case, the caller's FETCH will only return the 
	! receive parameter values once and perform no other processing.
	If NOT SqlFetchNext(hSqlCur1, nInd)
		Return 1
	Else
		Return 0
	Call SqlDisconnect(hSqlCur1)
\
,
/

Remark
\
This example shows how to generate a single row fetch from the database
and then manipulate that data in order to produce a multiple row result. 
In this case the output is only indirectly tied to the database - an
excellent means for producing what-if scenarios. In general, any fetches
from the caller do not necessarily have to have database sources within
the Procedure.
/
pause;

procedure: GUEST_WTPROC
Parameters
	String: sName
	Receive Number: nCurrentWT
	Receive Number: nDays
Local Variables
	Sql Handle: hSqlCur1
	String: sSelect 
	Number: nMaxWeight
	Number: nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		Set sSelect = 'select max(WEIGHT) \
			from GUEST_WT where NAME = :sName into :nMaxWeight'
		Call SqlPrepare(hSqlCur1, sSelect)
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
		! Because we also have an On Procedure Fetch state coded, 
		! this single row fetch will NOT be returned to the caller
		! and will only be used internally by the Procedure for
		! subsequent processing.
		Call SqlFetchNext(hSqlCur1, nInd)
		Set nCurrentWT = nMaxWeight
	On Procedure Fetch
		! List our weight by day (10lb daily increase) until we are 
		! ( > 200 pounds ) starting from our historical
		! maximum weight.
		! In this case, the caller will not be directly fetching 
		! from the database.
		If nCurrentWT < 200
			Set nCurrentWT = nCurrentWT + 10
			Set nDays = nDays + 1
			Return 0
		Else
			Return 1
	On Procedure Close
		Call SqlDisconnect(hSqlCur1)
\
DON JACKSON,,,
/

Remark
\
This example demonstrates the use of the Procedure trace command. The 
variable nCount will be traced and the output re-directed to a tracefile.
You can inspect this tracfile after this tutorial is run.
/
pause;

set tracefile TRACE.OUT;

procedure: P24
Local Variables
	Number: nCount
Actions
	Trace nCount
	Loop
		Set nCount = nCount + 1
		Trace nCount
		If nCount > 10
			Trace nCount
			Return 0
;

set tracefile off;

Remark
\
The following 2 examples demonstrate error handling as provided by the
SAL When SqlError construct.
/
pause;

-- A stored command is created and then exists in the database.
store WEIGHT_QUERY
select WEIGHT from GUEST_WT where NAME = :1;

Remark
\
However, in order to create the error condition, we need to drop the stored 
command just prior to Procedure example.
/
erase WEIGHT_QUERY;

Remark
\
In this example, the When SqlError is coded to trap error 207
and fix the problem of the missing stored command.  Use of Return TRUE
from When SqlError is shown.
/
pause;

procedure: GWPROC
Parameters
	String: sName
	Receive Number: nWeight
Local Variables
	Sql Handle: hSqlCur1
	Number: nInd
	Number: nRcd
Actions
	On Procedure Startup
		! This exception handling routine can detect the SQL error
		! generated by the SqlRetrieve call and can then proceed to
		! re-store the non-existent stored command.  In order to
		! continue processing the Procedure, we Return TRUE back
		! to the Procedure and execute the stored command. If we
		! encounter other SQL errors, no Return is executed and 
		! we immediately return (along with the SQL error code)
		! back to the caller.
		When SqlError
			Set nRcd = SqlError(hSqlCur1)
			If nRcd = 207
				Call SqlStore(hSqlCur1, 'WEIGHT_QUERY', \
					'select WEIGHT \
					from GUEST_WT \
					where NAME = :1 into :2')
				Call SqlCommit(hSqlCur1)
				Call SqlRetrieve(hSqlCur1, 'WEIGHT_QUERY', \
					':sName', ':nWeight')
				Return TRUE
		Call SqlConnect(hSqlCur1)

		! This call will fail due to the non-existent stored command.
		! In this instance, When SqlError will force SqlRetrieve to
		! return TRUE and the Procedure will continue to execute
		! successfully.
		Call SqlRetrieve(hSqlCur1, 'WEIGHT_QUERY', \
			':sName', ':nWeight')
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1)
\
DON JACKSON,,
/


Remark
\
In this example, the When SqlError construct tests for 2 error conditions:
	. stored command does NOT exist - error code 207.
	. table used in stored command does not exist - error code 601.
In this case, because the table required for the stored command was 
dropped prior to Procedure execution, the error code is 601. 

Returning FALSE (as opposed to executing no Return from When SqlError) 
allows the Procedure to provide additional processing such as returning 
a user defined error or a different error in error.sql (as in this example).

If the error code had been 207, the Procedure would re-store/retrieve the 
stored command and procedure execution would resume successfully.
/
pause;

-- To create the error condition, drop the table used in the stored command.
drop table GUEST_WT;

procedure: GWPROC
Parameters
	String: sName
	Receive Number: nWeight
Local Variables
	Sql Handle: hSqlCur1
	Number: nInd
	Number: nRcd
	Boolean: bCond
Actions
	On Procedure Startup
		! This exception handling routine Returns FALSE because
		! the required table for the stored command was dropped
		! prior to the execution of the procedure.
		When SqlError
			Set nRcd = SqlError(hSqlCur1)
			If nRcd = 207
				Call SqlStore(hSqlCur1, 'WEIGHT_QUERY', \
				   'select WEIGHT from GUEST_WT \
				 	where NAME = :1 into :2')
				Call SqlCommit(hSqlCur1)
				Call SqlRetrieve(hSqlCur1, 'WEIGHT_QUERY', \
				    ':sName', ':nWeight')
				Return TRUE
			Else If nRcd = 601
				Return FALSE
		Call SqlConnect(hSqlCur1)

		! When SqlError Returns FALSE and hence the return value
		! for bCond is set to FALSE.
		Set bCond = SqlRetrieve(hSqlCur1, 'WEIGHT_QUERY', \
			':sName', ':nWeight')

		! Since When SqlError set bCond to FALSE, we have a false 
		! condition - this will return control back to the caller 
		! with an error different from the original. Alternatively,
		! a user defined error (and a more meaningful error) could be 
		! created in ERROR.SQL and returned here.
		If NOT bCond
			Return 6302
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1)
\
JANE FYUDAI,,
/

-- Rollback the drop of table GUEST_WT.
rollback;

remark
\
This example shows the affect of inserting data and then retrieving that
data with subsequent select.
/
pause;

create table T1 (C1 integer, C2 integer);
insert into T1 values (1000, 2000);
commit;

procedure: P1
Parameters
	Receive Number: nOutput1
	Receive Number: nOutput2
Local Variables
	Sql Handle: hSqlCur1
	Sql Handle: hSqlCur2
	Number: nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		Call SqlConnect(hSqlCur2)
	On Procedure Execute
		Call SqlPrepare(hSqlCur1, 'Insert into T1 values (7,8)')
		Call SqlPrepare(hSqlCur2, 'Select C1, C2 \
			 from T1 into :nOutput1, :nOutput2')
		Call SqlExecute(hSqlCur1)
		Call SqlExecute(hSqlCur2)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur2, nInd)
			Return 1
		Else
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1)
		Call SqlDisconnect(hSqlCur2)
\
,,
/


Remark
\
=============================================================================
SECTION 2: STORED PROCEDURE EXAMPLES
=============================================================================
/
Remark
\
The following 2 examples contrast DYNAMIC and STATIC Procedures. DYNAMIC 
Procedures are the default and do NOT need to be stored as STATIC Procedures
do - however, DYNAMIC Procedures can optionally be stored as demonstrated
here.
/
pause;

-- Insert a GUEST with a weight over 200lbs into the GUEST_WT table.
insert into GUEST_WT values (:1,:2,:3)
\
MARLON SPANDEX,286,07-OCT-1992
/
commit;

Remark
\
In SQLTalk, Procedure receive parameter strings default to 80; hence the need 
to resize the first column generated from the procedure.
/
column 1 width 20;

Remark
\
This is an example of a DYNAMIC Procedure.  This cannot be a STATIC Procedure
due to the symbolic string substitution of the SQL statement found in the 
SqlPrepare() call. The SQL statements in a DYNAMIC Procedure are not 
pre-compiled and hence are not optimized or parsed when stored.
/
pause;

store DYNAMITE
procedure: DYNAMIC_SQL
Parameters
	Number: nOver200
	Receive String: sName
	Receive Number: nWeight
Local Variables
	Sql Handle: hSqlCur1
	String: sWhere
	String: sSelect
	Number: nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		! Check for weight over 200 pounds and build the appropiate 
		! where clause.
		If nOver200 = 1
			Set sWhere = 'where WEIGHT > 200'
		Else
			Set sWhere = 'where WEIGHT < 200'
		Set sSelect = 'select NAME, WEIGHT \
		        from GUEST_WT ' || sWhere || ' into :sName, :nWeight'
		Call SqlPrepare(hSqlCur1, sSelect)
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1)
/

execute DYNAMITE
\
1,,,
/


Remark
\
While STATIC Procedures don't provide the flexibility of a DYNAMIC
Procedure, they do optimize and parse SQL statements before storing
and hence yield higher performance at run-time.
/
pause;

store STATICS
procedure: STATIC_SQL static
Parameters
	Receive String: sName
	Receive Number: nWeight
Local Variables
	Sql Handle: hSqlCur1
	Number: nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		! After storing STATIC_SQL, SqlPrepare no longer performs the 
		! compile phase at run-time because the SQL statement is 
		! already compiled.
		! If the SQL statement has already been stored using SQLTalk
		! or SqlStore() - SqlRetrieve() can be substituted.
		Call SqlPrepare(hSqlCur1, 'select NAME, WEIGHT from \
			GUEST_WT into :sName, :nWeight')
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1)
/

execute STATICS
\
,,
/

-- Rollback Stored Procedures DYNAMITE & STATICS.
rollback;

-- Reset column 1 width to default.
column 1 width off;


Remark
\
DYNAMIC Procedure using SqlImmediate( ) within default On Procedure Execute
state. Update and an aggregate query performed. Notice that SqlFetchNext( )
is not required to fetch the row - this is because SqlImmediate( ) compiles,
executes, and for select statments can fetch a single row - but only a
single row.
/
pause;

create table CHECKING (ACCOUNTNUM number, BALANCE number);
insert into CHECKING values (1,1000);
commit;

store WITHDRAW
procedure: WITHDRAW
Parameters
	Number: nAccount
	Number: nAmount
	Receive Number: nNewBalance
Local Variables
	String: sUpdate
	String: sSelect
Actions
	Set sUpdate = 'update CHECKING \
		set BALANCE = BALANCE - :nAmount \
		where ACCOUNTNUM = :nAccount'
	Call SqlImmediate(sUpdate)
	Set sSelect = 'select BALANCE from CHECKING \
		where ACCOUNTNUM = :nAccount \
		into :nNewBalance'
	Call SqlImmediate(sSelect)
;

execute WithDraw
\
1,100,,
/
select * from CHECKING;

-- Rollback the Stored Procedure and update for the next example:
rollback;


Remark
\
In this example, DYNAMIC Procedure logic branches based on the
SqlImmediate( ) results supplied to the Receive Parameter nNewBalance.
/
pause;

store WITHDRAW
procedure: WITHDRAW
Parameters
	Number: nAccount
	Number: nAmount
	Receive Number: nNewBalance
	Receive Boolean: bOverDrawn
Local Variables
	String: sSelect
	String: sUpdate
Actions
	Set sSelect = 'select BALANCE from CHECKING \
 		where ACCOUNTNUM = :nAccount \
		into :nNewBalance'
	Call SqlImmediate(sSelect)
	Set nNewBalance = nNewBalance - nAmount
	If (nNewBalance < 0)
		Set bOverDrawn = TRUE
	Else
		Set bOverDrawn = FALSE
		Set sUpdate = 'update CHECKING \
		set BALANCE = BALANCE - :nAmount \
 			where	ACCOUNTNUM = :nAccount'
		Call SqlImmediate(sUpdate)
;

execute WITHDRAW
\
1,100,,,
/

select * from CHECKING;

-- Rollback the Stored Procedure and update for the next example:
rollback;

Remark
\
In this example, multiple cursors are opened. Procedure overhead is minimized
by placing compile calls into the On Procedure Startup section.
/
pause;

store WITHDRAW
procedure: WITHDRAW
Parameters
	Number: nAccount
	Number: nAmount
	Receive Number: nNewBalance
	Receive Boolean: bOverDrawn
Local Variables
	Sql Handle: hSelect
	Sql Handle: hUpdate
	String: sSelect
	String: sUpdate
	Number: nStatus
Actions
	On Procedure Startup
		Set sSelect = 'select BALANCE from CHECKING \
			where ACCOUNTNUM = :nAccount \
			into :nNewBalance'
		Set sUpdate = 'update CHECKING \
			set BALANCE = BALANCE - :nAmount \
 			where	ACCOUNTNUM = :nAccount'
		Call SqlConnect(hSelect)
		Call SqlPrepare(hSelect, sSelect)
		Call SqlConnect(hUpdate)
		Call SqlPrepare(hUpdate, sUpdate)
	On Procedure Execute
		Call SqlExecute(hSelect)
		Call SqlFetchNext(hSelect, nStatus)
		Set nNewBalance = nNewBalance - nAmount
		If (nNewBalance < 0)
			Set bOverDrawn = TRUE
		Else
			Set bOverDrawn = FALSE
			Call SqlExecute(hUpdate)
	On Procedure Close
		Call SqlDisconnect(hSelect)
		Call SqlDisconnect(hUpdate)
;

execute WITHDRAW
\
1,100,,,
/

select * from CHECKING;

-- Rollback the Stored Procedure and update for the next example:
rollback;

Remark
\
In this example, boolean returns are used in the On Procedure Fetch section.
/
pause;

store WITHDRAW
procedure: WITHDRAW
Parameters
	Number: nAccount
	Number: nAmount
	Receive Number: nNewBalance
Local Variables
	Sql Handle: hSelect
	String: sSelect
	Number: nStatus
	Boolean: bEOF
Actions
	On Procedure Startup
		Set sSelect = 'select BALANCE from CHECKING \
			where ACCOUNTNUM = :nAccount \
			into :nNewBalance'
		Call SqlConnect(hSelect)
		Call SqlPrepare(hSelect, sSelect)
	On Procedure Execute
		Call SqlExecute(hSelect)
		! Internal fetch - column value not returned to caller due to 
		! inclusion of the On Procedure Fetch state which DOES 
		! display values.
		Call SqlFetchNext(hSelect, nStatus)
	On Procedure Fetch
		If (nNewBalance > 100)
			Set nNewBalance = nNewBalance * 1.005
			Set nNewBalance = nNewBalance - 100
			Set bEOF = FALSE
		Else
			Set bEOF = TRUE
		Return bEOF
	On Procedure Close
		Call SqlDisconnect(hSelect)
/

execute WITHDRAW
\
1,100,,
/

select * from CHECKING;


Remark
\
=============================================================================
SECTION 3: NESTED PROCEDURE EXAMPLES
=============================================================================
/
Remark
\
This example shows how one Stored Procedure can call another Stored Procedure
- in this case the calling Stored Procedure is DYNAMIC and the called Stored
Procedure is STATIC.  Nesting Procedures can enhance the modularity of
Procedure code by creating common routines that perform specialized tasks
which can then be called by any number of different Procedures or
calling programs.
/
pause;

create table GUEST
(
NAME	varchar(25), 
SEX	varchar(1), 
BUILD	varchar(1), 
HEIGHT	integer
);

insert into GUEST values('BETTE MIDRIFF','F','M',66);

create table RATE
(
RATE	varchar(12), 
PER_DAY	double precision
);

insert into RATE values(:1, :2)
\
"HIGH ROLLER",300,
"BUSINESS",190,
"TOURIST",150,
"INDIGENT",25,
/

Remark
\
This STATIC Stored Procedure is called by the DYNAMIC Procedure
named ADJUST_RATE.  This Procedure determines the current population
of the GUEST table.
/
pause;

store GUEST_COUNT
procedure: GUEST_COUNT static
Parameters
	Receive Number: nCount
Local Variables
	Sql Handle: hSqlCur1
	Number: nInd
Actions
	Call SqlConnect( hSqlCur1 )
	Call SqlPrepareAndExecute( hSqlCur1, 'select count(*) from GUEST \
				into :nCount' )
	Call SqlFetchNext(hSqlCur1, nInd)
;

Remark
\
This Stored Procedure calls the GUEST_COUNT Stored Procedure. Based on
the current GUEST population, the DAILY rates are determined.
/
pause;

store ADJUST_RATE
procedure: ADJUST_RATE dynamic
Parameters
	Receive String: sRate
	Receive Number: nPerDay
Local Variables
	Sql Handle: hSqlCur1
	Number: nPop
	Number: nInd
	String: sAdjust
	String: sUpdate
	String: sSelect
Actions
	On Procedure Startup
		Call SqlConnect( hSqlCur1 )
		! Retrieve the Stored Procedure to get the current GUEST
		! count. Notice that the need to place the into variable also
		! into the bind list so that the Receive parameter will
		! be bound.
		Set nPop = 0
		Call SqlRetrieve( hSqlCur1, 'GUEST_COUNT', ':nPop', ':nPop' )
		Set sSelect = 'Select RATE, PER_DAY from RATE \
				into :sRate, :nPerDay'
	On Procedure Execute
		Call SqlExecute( hSqlCur1 )
		!
		! Fetch a single row value into the nPop local variable
		Call SqlFetchNext( hSqlCur1, nInd )
		!
		! Use dynamic SQL to build the update statement
		! based on the GUEST population.
		If nPop > 1
			Set sAdjust = 'set PER_DAY = PER_DAY * 1.15'
		Else
			Set sAdjust = 'set PER_DAY = PER_DAY * 1.05'
		Set sUpdate = 'Update RATE ' || sAdjust
		!
		! Update the RATE table
		Call SqlPrepareAndExecute( hSqlCur1, sUpdate )
		!
		! Now select the new rates from the RATE table
		Call SqlPrepareAndExecute( hSqlCur1, sSelect )
	On Procedure Fetch
		If NOT SqlFetchNext( hSqlCur1, nInd )
			Return 1
		Else
			Return 0
	On Procedure Close
		Call SqlDisconnect( hSqlCur1 )
;

-- Resize string column.
column 1 width 15;


execute ADJUST_RATE
\
,,
/

-- Reset column width to default.
column 1 width off;

REMARK
\
This example illustrates the ability to nest multiple Procedures which return
multiple rows. In this case, the base Procedure is LEVEL1PROC, which is
called by LEVEL2PROC, which is called by LEVEL3PROC, ... etc. Multiple rows
are returned to the top caller via each Procedure's 'On Procedure Fetch'
section.
The data being returned is also massaged by every level in order to show
how the individual rows can be manipulated as they pass through the levels.
The last successful execution is either LEVEL8PROC, LEVEL9PROC or LEVEL10PROC
- on some operating systems a maximum nesting limit may be exceeded and displayed 
as error "stack-overflow in processing request" or equivalent.
/
pause;

create table LEVEL1DATA (COL1 integer, COL2 varchar(20), COL3 date);

insert into LEVEL1DATA values (:1, :2, :3)
\
1, 'level1 data', 01-JAN-1995
2, 'level1 data', 02-JAN-1995
3, 'level1 data', 03-JAN-1995
4, 'level1 data', 04-JAN-1995
5, 'level1 data', 05-JAN-1995
6, 'level1 data', 06-JAN-1995
7, 'level1 data', 07-JAN-1995
8, 'level1 data', 08-JAN-1995
9, 'level1 data', 09-JAN-1995
/

store LEVEL1PROC
procedure: LEVEL1PROC static
Parameters
	Receive Number:	nLvl1Num
	Receive String:	sLvl1Str
	Receive Date/Time: dtLvl1Date
Local Variables
	Sql Handle: hSqlCur1
	Number:	nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		Call SqlPrepare(hSqlCur1, 'select COL1, COL2, COL3 \
		     from LEVEL1DATA into :nLvl1Num, :sLvl1Str, :dtLvl1Date')
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
		    	Return 1
		Else
			Set nLvl1Num = nLvl1Num + 1
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1);

store LEVEL2PROC
procedure: LEVEL2PROC static
Parameters
	Receive Number:	nLvl2Num
	Receive String:	sLvl2Str
	Receive Date/Time: dtLvl2Date
Local Variables
	Sql Handle: hSqlCur1
	Number:	nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		Call SqlRetrieve(hSqlCur1, 'LEVEL1PROC', \
			':nLvl2Num, :sLvl2Str, :dtLvl2Date', \
			':nLvl2Num, :sLvl2Str, :dtLvl2Date')
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Set nLvl2Num = nLvl2Num + 1
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1);

store LEVEL3PROC
procedure: LEVEL3PROC static
Parameters
	Receive Number:	nLvl3Num
	Receive String:	sLvl3Str
	Receive Date/Time: dtLvl3Date
Local Variables
	Sql Handle: hSqlCur1
	Number:	nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		Call SqlRetrieve(hSqlCur1, 'LEVEL2PROC', \
			':nLvl3Num, :sLvl3Str, :dtLvl3Date', \
			':nLvl3Num, :sLvl3Str, :dtLvl3Date')
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Set nLvl3Num = nLvl3Num + 1
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1);

store LEVEL4PROC
procedure: LEVEL4PROC static
Parameters
	Receive Number:	nLvl4Num
	Receive String:	sLvl4Str
	Receive Date/Time: dtLvl4Date
Local Variables
	Sql Handle: hSqlCur1
	Number:	nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		Call SqlRetrieve(hSqlCur1, 'LEVEL3PROC', \
			':nLvl4Num, :sLvl4Str, :dtLvl4Date', \
			':nLvl4Num, :sLvl4Str, :dtLvl4Date')
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Set nLvl4Num = nLvl4Num + 1
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1);

store LEVEL5PROC
procedure: LEVEL5PROC static
Parameters
	Receive Number:	nLvl5Num
	Receive String:	sLvl5Str
	Receive Date/Time: dtLvl5Date
Local Variables
	Sql Handle: hSqlCur1
	Number:	nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		Call SqlRetrieve(hSqlCur1, 'LEVEL4PROC',\
			':nLvl5Num, :sLvl5Str, :dtLvl5Date', \
			':nLvl5Num, :sLvl5Str, :dtLvl5Date')
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Set nLvl5Num = nLvl5Num + 1
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1);

store LEVEL6PROC
procedure: LEVEL6PROC static
Parameters
	Receive Number:	nLvl6Num
	Receive String:	sLvl6Str
	Receive Date/Time: dtLvl6Date
Local Variables
	Sql Handle: hSqlCur1
	Number:	nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		Call SqlRetrieve(hSqlCur1, 'LEVEL5PROC', \
			':nLvl6Num, :sLvl6Str, :dtLvl6Date', \
			':nLvl6Num, :sLvl6Str, :dtLvl6Date')
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Set nLvl6Num = nLvl6Num + 1
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1);

store LEVEL7PROC
procedure: LEVEL7PROC static
Parameters
	Receive Number:	nLvl7Num
	Receive String:	sLvl7Str
	Receive Date/Time: dtLvl7Date
Local Variables
	Sql Handle: hSqlCur1
	Number:	nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		Call SqlRetrieve(hSqlCur1, 'LEVEL6PROC', \
			':nLvl7Num, :sLvl7Str, :dtLvl7Date', \
			':nLvl7Num, :sLvl7Str, :dtLvl7Date')
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Set nLvl7Num = nLvl7Num + 1
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1);

store LEVEL8PROC
procedure: LEVEL8PROC static
Parameters
	Receive Number:	nLvl8Num
	Receive String:	sLvl8Str
	Receive Date/Time: dtLvl8Date
Local Variables
	Sql Handle: hSqlCur1
	Number:	nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		Call SqlRetrieve(hSqlCur1, 'LEVEL7PROC', \
			':nLvl8Num, :sLvl8Str, :dtLvl8Date', \
			':nLvl8Num, :sLvl8Str, :dtLvl8Date')
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Set nLvl8Num = nLvl8Num + 1
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1);

store LEVEL9PROC
procedure: LEVEL9PROC static
Parameters
	Receive Number:	nLvl9Num
	Receive String:	sLvl9Str
	Receive Date/Time: dtLvl9Date
Local Variables
	Sql Handle: hSqlCur1
	Number:	nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		Call SqlRetrieve(hSqlCur1, 'LEVEL8PROC', \
			':nLvl9Num, :sLvl9Str, :dtLvl9Date', \
			':nLvl9Num, :sLvl9Str, :dtLvl9Date')
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Set nLvl9Num = nLvl9Num + 1
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1);

store LEVEL10PROC
procedure: LEVEL10PROC static
Parameters
	Receive Number:	nLvl10Num
	Receive String:	sLvl10Str
	Receive Date/Time: dtLvl10Date
Local Variables
	Sql Handle: hSqlCur1
	Number:	nInd
Actions
	On Procedure Startup
		Call SqlConnect(hSqlCur1)
		Call SqlRetrieve(hSqlCur1, 'LEVEL9PROC', \
			':nLvl10Num, :sLvl10Str, :dtLvl10Date', \
			':nLvl10Num, :sLvl10Str, :dtLvl10Date')
	On Procedure Execute
		Call SqlExecute(hSqlCur1)
	On Procedure Fetch
		If NOT SqlFetchNext(hSqlCur1, nInd)
			Return 1
		Else
			Set nLvl10Num = nLvl10Num + 1
			Return 0
	On Procedure Close
		Call SqlDisconnect(hSqlCur1);

-- Resize string column.
column 2 width 13;

execute LEVEL1PROC
\
,,,
/

execute LEVEL2PROC
\
,,,
/

execute LEVEL3PROC
\
,,,
/

execute LEVEL4PROC
\
,,,
/

execute LEVEL5PROC
\
,,,
/

execute LEVEL6PROC
\
,,,
/

execute LEVEL7PROC
\
,,,
/

execute LEVEL8PROC
\
,,,
/

-- Maximum nesting level may be exceeded on some operating systems.
execute LEVEL9PROC
\
,,,
/

-- Maximum nesting level may be exceeded on some operating systems.
execute LEVEL10PROC
\
,,,
/

-- Rollback the above stored procedures.
rollback;

-- Reset column width to default.
column 2 width off;

-- Drop all prior committed objects in advance of creating data for SP.APP.
drop table GUEST_WT;
drop table T1;
drop table CHECKING;
erase WEIGHT_QUERY;
commit;

Remark
\
============================================================================
SECTION 4: REQUIRED DATA AND DATABASE OBJECTS FOR SP.APP
============================================================================
/
Remark
\
Create the following tables, data and Stored Procedures within the ISLAND 
database so that the SQLWindows Stored Procedure application SP.APP can
be demonstrated.
/
pause;

-- Turn pause off to expedite creation of tables and data.
set pause off;

create table GUEST
(
NAME varchar(25) not null, 
BUILD varchar(1), 
HEIGHT integer, 
SEX varchar(1), 
CHECK_IN date not null, 
CHECK_OUT date, 
IN_WEIGHT decimal(3, 0), 
TARGET_WEIGHT decimal(3, 0), 
ROOM_NAME varchar(20), 
TRAINER varchar(10)
);

insert into GUEST values(:1,:2,:3,:4,:5,:6,:7,:8,:9,:10)
\
HEATHER STARLETTE,M,68,F,1992-10-10,1993-02-04,180,155,PUCE RM,SERENA,
BIANCA JOGGER,S,61,F,1992-10-05,,120,110,BEIGE RM,TODD,
STELLA SHIELDS,L,65,F,1992-10-19,,122,117,PUCE RM,JULIO,
ENGLEBERT HUMPTY-DUMPTY,S,72,M,1992-10-22,,167,180,PEARL SUITE,,
SHIRLEY O'SHADE,M,64,F,1992-10-19,,140,130,SAND RM,TODD,
MEL GIMLET,M,70,M,1992-10-23,,210,175,PEARL SUITE,KATHY,
JOAN TONIC,S,62,F,1992-10-05,,150,130,PEARL SUITE,CANDY,
DOLLY BRISKET,S,61,F,1992-10-10,,130,115,PEACH RM,TODD,
JANE FYUNDAI,S,65,F,1992-10-12,,132,125,GOLD RM,,
DON JACKSON,M,70,M,1992-10-10,,176,165,PEARL SUITE,JULIO,
DYAN HOWITZER,S,68,F,1992-10-05,,105,120,AMBER RM,JULIO,
MARCELLO HISTRIONI,M,66,M,1992-10-06,,210,180,BEIGE RM,KATHY,
JEAN-PAUL ROTUNDO,S,70,M,1992-10-05,,200,165,PUCE RM,JULIO,
MICHAEL JOHNSON,L,71,M,1992-10-14,,198,180,PEARL SUITE,SERENA,
CLINT WESTWOOD,M,73,M,1992-10-04,,170,180,INDIGO RM,,
JOANNIE RIVULETS,L,68,F,1992-10-04,,175,150,ANNATTO RM,SERENA,
WARREN AMOROSO,L,70,M,1992-10-06,,200,180,PEARL SUITE,,
MARLON SPANDEX,L,71,M,1992-10-06,,287,225,PEACH RM,KATHY,
OLYMPIA WELCH,L,65,F,1992-10-17,1993-01-17,200,170,AMBER RM,MICHAEL,
BETTE MIDRIFF,S,66,F,1992-10-10,,160,101,AMBER RM,JULIO,
BOZO,S,72,M,1992-10-20,,300,187,MARINE RM,,
SEAN PENCIL,M,65,M,1992-10-04,,172,155,INDIGO RM,,
/

create unique index GUESTIX on GUEST(NAME);

create table GUEST_TREATMENT
(NAME		varchar(25) not null, 
TREATMENT	varchar(12), 
START_DATE	date, 
END_DATE	date, 
PRICE		decimal(6, 2)
);

insert into GUEST_TREATMENT values(:1,:2,:3,:4,:5)
\
BETTE MIDRIFF,WEIGHTS,1992-10-11,1992-10-18,100,
BETTE MIDRIFF,JOGGING,1992-10-11,1992-10-18,0,
BIANCA JOGGER,ACUPUNCTURE,1992-10-06,1992-10-13,150,
MARLON SPANDEX,MEDITATION,1992-10-07,1992-10-14,60,
MICHAEL JOHNSON,ACUPUNCTURE,1992-10-15,1992-10-22,150,
MARLON SPANDEX,KARATE,1992-10-07,1992-10-14,150,
JANE FYUNDAI,KARATE,1992-10-13,1992-10-20,150,
JANE FYUNDAI,SUMO DIET,1992-10-13,1992-10-20,120,
DOLLY BRISKET,WEIGHTS,1992-10-11,1992-10-18,100,
DOLLY BRISKET,MEDITATION,1992-10-11,1992-10-18,60,
CLINT WESTWOOD,PAINT BALL,1992-10-05,1992-10-12,20,
BIANCA JOGGER,RAQUETBALL,1992-10-06,1992-10-13,40,
BOZO,TENNIS,1992-10-21,1992-10-28,150,
CLINT WESTWOOD,TENNIS,1992-10-05,1992-10-12,150,
ENGLEBERT HUMPTY-DUMPTY,PAINT BALL,1992-10-23,1992-10-30,20,
HEATHER STARLETTE,BALLET,1992-10-11,1992-10-18,150,
JEAN-PAUL ROTUNDO,SUMO DIET,1992-10-06,1992-10-13,120,
JOAN TONIC,JOGGING,1992-10-06,1992-10-13,0,
JOANNIE RIVULETS,BALLET,1992-10-05,1992-10-12,150,
MARCELLO HISTRIONI,SQUASH,1992-10-07,1992-10-14,20,
MEL GIMLET,SUMO DIET,1992-10-24,1992-10-31,120,
MICHAEL JOHNSON,AROMATHERAPY,1992-10-15,1992-10-22,100,
OLYMPIA WELCH,JOGGING,1992-10-18,1992-10-25,0,
SEAN PENCIL,PAINT BALL,1992-10-05,1992-10-12,20,
BOZO,WEIGHTS,1992-10-21,1992-10-28,100,
CLINT WESTWOOD,SQUASH,1992-10-05,1992-10-12,20,
DOLLY BRISKET,BOWLING,1992-10-11,1992-10-18,10,
DYAN HOWITZER,WEIGHTS,1992-10-06,1992-10-13,100,
ENGLEBERT HUMPTY-DUMPTY,SWIMMING,1992-10-23,1992-10-30,60,
HEATHER STARLETTE,AROMATHERAPY,1992-10-11,1992-10-18,100,
JEAN-PAUL ROTUNDO,MEDITATION,1992-10-06,1992-10-13,60,
JOAN TONIC,BOWLING,1992-10-06,1992-10-13,10,
JOANNIE RIVULETS,JOGGING,1992-10-05,1992-10-12,0,
MARCELLO HISTRIONI,KARATE,1992-10-07,1992-10-14,150,
MEL GIMLET,SWIMMING,1992-10-24,1992-10-31,60,
MICHAEL JOHNSON,KARATE,,,150,
OLYMPIA WELCH,MEDITATION,1992-10-18,1992-10-25,60,
STELLA SHIELDS,PAINT BALL,1992-10-20,1992-10-27,20,
WARREN AMOROSO,SWIMMING,1992-10-07,1992-10-14,60,
BETTE MIDRIFF,AEROBICS,1992-10-11,1992-10-18,40,
BIANCA JOGGER,KARATE,1992-10-06,1992-10-13,150,
BOZO,BOWLING,1992-10-21,1992-10-28,10,
CLINT WESTWOOD,KARATE,1992-10-05,1992-10-12,150,
DOLLY BRISKET,SQUASH,1992-10-11,1992-10-18,20,
DON JACKSON,KARATE,1992-10-11,1992-10-18,150,
DYAN HOWITZER,KARATE,1992-10-06,1992-10-13,150,
ENGLEBERT HUMPTY-DUMPTY,ICE CREAM,1992-10-23,1992-10-30,500,
HEATHER STARLETTE,MEDITATION,1992-10-11,1992-10-18,60,
JANE FYUNDAI,AEROBICS,1992-10-13,1992-10-20,40,
JOAN TONIC,ACUPUNCTURE,1992-10-06,1992-10-13,150,
JOANNIE RIVULETS,KARATE,1992-10-05,1992-10-12,150,
MARCELLO HISTRIONI,MEDITATION,1992-10-07,1992-10-14,60,
MARLON SPANDEX,SWIMMING,1992-10-07,1992-10-14,60,
MICHAEL JOHNSON,KARATE,1992-10-15,1992-10-22,150,
OLYMPIA WELCH,BALLET,1992-10-18,1992-10-25,150,
SHIRLEY O'SHADE,SWIMMING,1992-10-20,1992-10-27,60,
STELLA SHIELDS,KARATE,1992-10-20,1992-10-27,150,
WARREN AMOROSO,BALLET,1992-10-07,1992-10-14,150,
/

-- Set pause back on so that the SP.APP Stored Procedures can be reviewed.
set pause on;

Remark
\
These are the Stored Procedures required for SP.APP. They are executed
here to demonstrate their operation.
/
pause;

store GUEST_SP
procedure: GUEST_SP static
Parameters
	Receive String: sName
Local Variables
	Sql Handle: hSqlCur1
	Number:	nInd
Actions
	On Procedure Execute
		Call SqlConnect( hSqlCur1 )
		Call SqlPrepareAndExecute ( hSqlCur1, 'select NAME \
				     into :sName from GUEST order by NAME' )
	On Procedure Fetch
		If NOT SqlFetchNext( hSqlCur1, nInd )
			Return 1
		Else
			Return 0
	On Procedure Close
		Call SqlDisconnect( hSqlCur1 )
;

-- Resize string column.
column 1 width 25;

execute GUEST_SP
\
,
/

store GUEST_SP2
procedure: GUEST_SP2 
Parameters
	String: sName
	Receive String: sBuild
	Receive Number: nHeight
	Receive String: sRoom
Local Variables
	Sql Handle: hSqlCur1
	String: sSelect
	Number:	nInd
Actions
	On Procedure Startup
		Call SqlConnect( hSqlCur1 )
		Set sSelect = 'select BUILD, HEIGHT, ROOM_NAME \
			       from GUEST where NAME = :sName
			       into :sBuild, :nHeight, :sRoom'
		Call SqlPrepare( hSqlCur1, sSelect )
	On Procedure Execute
		Call SqlExecute( hSqlCur1 )
	On Procedure Fetch
		If NOT SqlFetchNext( hSqlCur1, nInd)
			Return 1
		Else
			Return 0
	On Procedure Close
		Call SqlDisconnect( hSqlCur1 )
;

-- Resize string columns.
column 1 width 1;
column 3 width 20;

execute GUEST_SP2
\
MEL GIMLET,,,,
/

-- Resize string column.
column 1 width 12;

store TREATMENT_SP
procedure: TREATMENT_SP dynamic
Parameters
	String: sName
	Receive String: sTreatment
	Receive Date/Time: dtStart
	Receive Date/Time: dtEnd
	Receive Number: nPrice
Local Variables
	Sql Handle: hSqlCur1
	String: sSelect
	Number:	nInd
Actions
	On Procedure Startup
		Call SqlConnect( hSqlCur1 )
		Set sSelect = 'select TREATMENT, START_DATE, END_DATE, PRICE \
			    from GUEST_TREATMENT where NAME = :sName
			    into :sTreatment, :dtStart, :dtEnd, :nPrice'
		Call SqlPrepare( hSqlCur1, sSelect )
	On Procedure Execute
		Call SqlExecute( hSqlCur1 )
	On Procedure Fetch
		If NOT SqlFetchNext( hSqlCur1, nInd)
			Return 1
		Else
			Return 0
	On Procedure Close
		Call SqlDisconnect( hSqlCur1 )
;

-- Resize string column.
column 1 width 12;

execute TREATMENT_SP
\
MEL GIMLET,,,,,
/

commit;

-- Reset column widths to default.
column 1 width off;
column 3 width off;

-- 
-- END OF SP.SQL
--