remark
\
It is suggested you view this tutorial in a maximized window
if you are using SQLTalk for Windows.  To execute this 
tutorial, connect to the DEMO database using SQLTalk and
issue the run command from the prompt: SQL> RUN TRIGGERS.SQL;
/
pause;

REMARK
\
This script is intended to illustrate the capabilities of the SQLBase SQL
language Triggers.

Procedures and Triggers are often used to implement business rules that could
not be implemented at the database server with standard SQL referential
integrity.

A Stored Procedure is a sequence of SQLWindows Application Language (SAL)
statements that is assigned a name, compiled, and stored into a SQLBase
database.  Once the Stored Procedure has been defined in the database, an
application program can retrieve it by name and execute it.  This application
could be a C program, a SQLWindows program, or even an ad hoc query facility
like SQLTalk.
/
PAUSE
/
REMARK
\
This script will show the implementation of some very simple business rules
as they apply to a fictitious company's payroll system.  We will implement
the following business rules:

     (1) An employee cannot receive a salary raise exceeding
	 twenty percent of the employee's current salary.

     (2) All employee salaries must remain within the salary
	 range for the employee's position.

First, let's setup the appropriate tables.  The first table is the employee
table and will contain information specific to each employee.  Notice the
number of columns are kept to a minimum for simplicity.
/
CREATE TABLE EMP (EMP_NO     INTEGER,
		  EMP_NAME   VARCHAR(18),
		  EMP_SALARY DECIMAL(8,2),
		  EMP_JOB_NO INTEGER)
/
REMARK
\
Now, let's create a JOB table containing info specific to each company job.
/
CREATE TABLE JOB (JOB_NO	 INTEGER,
		  JOB_DESC	 VARCHAR(18),
		  JOB_MIN_SALARY DECIMAL(8,2),
		  JOB_MAX_SALARY DECIMAL(8,2))
/
REMARK
\
Let's populate the JOB table with some sample data.
/
INSERT INTO JOB VALUES (:1,:2,:3,:4)
\
101,Senior Programmer,50000,65000
102,Programmer,40000,55000
103,Junior Programmer,30000,45000
/
REMARK
\
Let's review what we have entered in the JOB table
/
SELECT * FROM JOB ORDER BY JOB_NO
/
PAUSE
/
REMARK
\
Ok.  Now we need to store a Procedure which enforces the first of our two
business rules.  First, we need to store a Procedure to prevent an employee
from receiving a salary raise of more than twenty percent.

This Stored Procedure will be called SALARY_RULE1 and requires two input
parameters, namely, the employee's old salary and the employee's new salary.
The salary raise rule is checked and if it violates the rule we return a
value of 6302 which corresponds to SQLBase error code 6302 defined in 
ERROR.SQL as "The value of an input variable is too large for the target 
column".  You could also create your own customized error code and message.
See the SQLBase SQL Language manual on how to create your own customized 
error messages.
/
STORE SALARY_RULE1
PROCEDURE: SALARY_RULE1 STATIC
  PARAMETERS
    NUMBER: nOldSalary
    NUMBER: nNewSalary
  ACTIONS
    If nNewSalary > nOldSalary + nOldSalary * .20
      Return 6302
    Else
      Return 0
/
REMARK
\
Now we need to store a Procedure to enforce the second of our two business
rules.	This second rule requires all employee salaries to remain within the
salary range for the employee's job.

This Stored Procedure will be called SALARY_RULE2 and requires two input
parameters, namely, the employee's job number and salary.  Local variables
are used to define a SQL handle, a fetch status code, and the minimum and 
maximum allowable salary for the specified employee's job.  The Stored 
Procedure connects to the database and selects the minimum and maximum 
allowable salaries for the specified job. If the specified salary falls
within the allowable range then the Stored Procedure ends gracefully by 
returning zero.  If the specified salary does not fall within the acceptable
salary range, then error 6302 is returned.
/
PAUSE
/
STORE SALARY_RULE2
PROCEDURE: SALARY_RULE2 STATIC
  PARAMETERS
    NUMBER:	nJob
    NUMBER:	nSalary
  LOCAL VARIABLES
    SQL HANDLE: hSql
    NUMBER:	nFetchStatus
    NUMBER:	nMax
    NUMBER:	nMin
  ACTIONS
    Call SqlConnect(hSql)
    Call SqlPrepare(hSql, 'SELECT JOB_MAX_SALARY, JOB_MIN_SALARY FROM JOB\
      WHERE JOB_NO = :nJob INTO :nMax, :nMin')
    Call SqlExecute(hSql)
    Call SqlFetchNext(hSql, nFetchStatus)
    Call SqlDisconnect(hSql)
    If nSalary < nMin
      Return 6302
    Else if nSalary > nMax
      Return 6302
    Else
      Return 0
/
REMARK
\
Now we will create a Trigger so an employee's salary is verified whenever a
salary change occurs regardless of who changes it or how it changed.  In
other words, one of the real beauties of a Trigger is that it removes the 
logic from the application program and places the logic in the database
server.  Whenever an update of a salary occurs, this Trigger will invoke an 
inline Procedure to check that our two business rules are not violated.  The 
inline Procedure executes the two Stored Procedures SALARY_RULE1 and 
SALARY_RULE2.  This Trigger executes the inline Procedure before the update 
of a salary and references the old employee's salary and the new employee's 
salary as input parameters to the SALARY_RULE1 Stored Procedure. This Trigger 
is activated at UPDATE time for each row of the EMP table.
/
PAUSE
/
CREATE TRIGGER SALARY_UPDT BEFORE UPDATE OF EMP_SALARY ON EMP
  REFERENCING OLD AS OLD_EMP NEW AS NEW_EMP
  (EXECUTE INLINE (OLD_EMP.EMP_SALARY, NEW_EMP.EMP_SALARY, NEW_EMP.EMP_JOB_NO)
  PROCEDURE: SALARY_UPDT STATIC
  PARAMETERS
    NUMBER:	nOldSalary
    NUMBER:	nNewSalary
    NUMBER:	nJob
  LOCAL VARIABLES
    SQL HANDLE: hSql
  ACTIONS
    On Procedure Startup
      Call SqlConnect(hSql)
    On Procedure Execute
      Call SqlRetrieve(hSql,'SALARY_RULE1',':nOldSalary,:nNewSalary',strNULL)
      Call SqlExecute(hSql)
      Call SqlRetrieve(hSql,'SALARY_RULE2',':nJob,:nNewSalary',strNULL)
      Call SqlExecute(hSql)
    On Procedure Close
      Call SqlDisconnect(hSql)
  )
 FOR EACH ROW
/
REMARK
\
Let's create the Triggers to verify salaries remain within the specified
salary ranges for each job.  This first Trigger to check the specified salary 
and job during the insertion of a new employee.
/
CREATE TRIGGER EMP_ISRT BEFORE INSERT ON EMP
  (EXECUTE SALARY_RULE2 (EMP.EMP_JOB_NO, EMP.EMP_SALARY))
  FOR EACH ROW
/
REMARK
\
This next Trigger checks the salary of an employee who are changing jobs.
We want to verify employee's salary are within the salary range of the newly
assigned job.
/
CREATE TRIGGER JOB_UPDT BEFORE UPDATE OF EMP_JOB_NO ON EMP
  REFERENCING OLD AS OLD_EMP NEW AS NEW_EMP
  (EXECUTE SALARY_RULE2 (NEW_EMP.EMP_JOB_NO, NEW_EMP.EMP_SALARY))
  FOR EACH ROW
/
REMARK
\
Let's do some testing and see how the Procedures and Triggers work together.

First, let's add a junior programmer to our staff but mistakenly his job
offer was for a salary of $25,000!   Of course, this should not be allowed as 
junior programmers have a salary range of $30,000 to $45,000.

The Trigger EMP_ISRT invokes the Stored Procedure SALARY_RULE2 to verify
that the salary is valid for the specified job.
/
INSERT INTO EMP VALUES (1, 'Bill Bates', 25000, 103)
/
REMARK
\
We inform Bill Bates of our decision to start him at $30,000.
Bill is excited.
/
INSERT INTO EMP VALUES (1, 'Bill Bates', 30000, 103)
/
REMARK
\
Well, its been several years and although Bill Bates can't seem to deliver
his software on time, we figure its time to give him a raise.  So, we decide 
to give Bill a new salary of $35,000.  Bill is excited.  This should be 
alright because it does not violate either of our two business rules.  The 
raise is less than 20 percent and his salary should still be within the 
salary range of a programmer.
/
UPDATE EMP SET EMP_SALARY = 35000 WHERE EMP_NO = 1
/
REMARK
\
So, what information do we have on Bill Bates in the employee table?
/
SELECT EMP_NO, EMP_NAME, EMP_SALARY, JOB_NO, JOB_DESC FROM EMP, JOB
WHERE EMP_NO = 1 AND EMP_JOB_NO = JOB_NO
/
REMARK
\
Several years pass.  We're still waiting for Bill's software.  Bill's been
a junior programmer and at the same salary for quite awhile.  Management
decides maybe a promotion will help Bill become more motivated.  So, 
management decides to tell Bill he has been promoted from a junior programmer
to a programmer.  Bill is excited.  Unfortunately, promoting Bill to a 
programmer without giving him an approriate salary increase is not allowed.

The Trigger JOB_UPDT invokes the Stored Procedure SALARY_RULE2 to verify that
the salary is valid for the specified job.
/
UPDATE EMP SET EMP_JOB_NO = 102 WHERE EMP_NO = 1
/
REMARK
\
So, management decides to give Bill a promotion AND a raise to $40,000 (the
minimum salary allowed for a programmer).  Bill is really excited.
/
UPDATE EMP SET EMP_JOB_NO = 102, EMP_SALARY = 40000 WHERE EMP_NO = 1
/
REMARK
\
So, what information do we have on Bill Bates in the employee table?
/
SELECT EMP_NO, EMP_NAME, EMP_SALARY, JOB_NO, JOB_DESC FROM EMP, JOB
WHERE EMP_NO = 1 AND EMP_JOB_NO = JOB_NO
/
REMARK
\
A miracle!  Bill finishes his software project.  Bill is excited.  Management
is excited.  Everyone is excited.  Management decides to push Bill's salary 
up to the max for a programmer, $55,000!  But, poor Bill. A raise of this 
amount will exceed a 20 percent increase and violates our first business
rule.

The Trigger SALARY_UPDT invokes the Stored Procedure SALARY_RULE1 to verify
that a salary raise does not exceed 20 percent.
/
UPDATE EMP SET EMP_SALARY = 55000 WHERE EMP_NO = 1
/
REMARK
\
Management explains to Bill the business policy but insist they'll increase
his salary now to $48,000 and then in six months, another $8,000 increase to
$56,000.  This first raise is allowed.
/
UPDATE EMP SET EMP_SALARY = 48000 WHERE EMP_NO = 1
/
REMARK
\
So, what information do we have on Bill Bates in the employee table?
/
SELECT EMP_NO, EMP_NAME, EMP_SALARY, JOB_NO, JOB_DESC FROM EMP, JOB
WHERE EMP_NO = 1 AND EMP_JOB_NO = JOB_NO
/
REMARK
\
But, the second raise is not allowed because a salary of $56,000 will push
Bill's salary above the allowable maximum for a programmer ($55,000).

The Trigger SALARY_UPDT invokes the Stored Procedure SALARY_RULE2 to verify
that a salaries are within the allowable ranges for the specified job.
/
UPDATE EMP SET EMP_SALARY = 56000 WHERE EMP_NO = 1
/
REMARK
\
Therefore, management decides to give Bill the raise and a promotion to 
senior programmer before he retires.
/
UPDATE EMP SET EMP_JOB_NO = 101, EMP_SALARY = 56000 WHERE EMP_NO = 1
/
REMARK
\
So, what information do we have on Bill Bates in the employee table?
/
SELECT EMP_NO, EMP_NAME, EMP_SALARY, JOB_NO, JOB_DESC FROM EMP, JOB
WHERE EMP_NO = 1 AND EMP_JOB_NO = JOB_NO
/

-- 
-- END OF TRIGGERS.SQL
--